# Query Patterns

This file contains advanced search strategies and multi-step workflows for the Tango MCP server.

## Important: Tool Selection Guide

**`search_contracts`** - Use for AWARDED contracts from FPDS (past awards)
- Competitive intelligence (who won what)
- Market sizing and trends
- Recompete pipeline (expiring contracts)
- Vendor portfolio analysis

**`search_opportunities`** - Use for ACTIVE solicitations from SAM.gov (current bids)
- Live bidding opportunities
- Response deadline tracking
- Sources sought and pre-solicitations

**`search_forecasts`** - Use for PLANNED opportunities (future pipeline)
- Pipeline planning before RFPs hit SAM.gov
- Note: VA does NOT publish forecasts to Tango
- HHS has 165+ forecasts in software services NAICS codes

## New Tool Patterns

### IDV Analysis (Vehicle Search)
Find active IDIQs or GWACs at VA to understand procurement vehicles.
```javascript
search_idvs({
  awarding_agency: "Department of Veterans Affairs",
  idv_type: "B", // IDC/IDIQ
  naics_code: ["541512"],
  expiring_after: "2025-10-01", // Must be active next FY
  limit: 20
})
```

### Forecast Planning (HHS Example)
Look for upcoming opportunities before they hit SAM.gov.
```javascript
search_forecasts({
  agency: "HHS",
  naics_code: ["541512"],
  fiscal_year: 2025,
  status: "PUBLISHED"
})
```

**WARNING:** VA does NOT publish forecasts to Tango. All 49 VA sub-agencies show `has_forecasts: false`. Use `search_opportunities` for VA pipeline visibility instead.

### Rapid Competitor Recon
Get a quick AI summary of a potential teaming partner or competitor.
```javascript
get_company_intelligence({
  company_name: "Oddball"
})
```
**Note:** Use exact legal company names for best results. Name matching is imprecise. Returns AI-generated summary, recent news, and key contacts.

### Full Vendor Profile with History
Get comprehensive SAM.gov data including certifications, federal obligations, and recent contract history.
```javascript
get_vendor_profile({
  uei: "UP4JHKJQ9MQ7",  // A6 UEI (or any competitor UEI)
  include_history: true,
  history_limit: 10
})
```
**Returns:** Business types, certifications, points of contact, federal_obligations (active/total contracts and spending), and recent contract/grant awards. Use for due diligence, capability assessment, and tracking competitor activity.

## Standard Patterns
For basic patterns, see the main SKILL.md.

## Core Search Patterns

### Pattern 1: Daily VA Opportunity Discovery

**Purpose:** Monitor active VA solicitations matching A6's profile

```javascript
search_opportunities({
  agency: "Department of Veterans Affairs",
  naics_code: ["541512", "541511", "541519"],
  set_aside_type: "SDVOSB",
  posted_date_after: "YYYY-MM-DD",       // Last check date
  active: true,
  ordering: "-posted_date",
  limit: 50
})
```

**Analysis Workflow:**
1. Run query daily or weekly
2. Check descriptions against PURSUE criteria:
   - VA.gov keywords
   - Forms digitization
   - Benefits processing
   - Accessibility requirements
   - Agile/HCD methodology
3. Flag response deadlines
4. Add high-priority opportunities to pipeline

**Variations:**
- Remove `set_aside_type` to see full & open competitions
- Adjust `posted_date_after` based on frequency of checks

### Pattern 2: SDVOSB Competitive Landscape (Contract Awards)

**Purpose:** Analyze the competitive SDVOSB landscape at VA via past awards

```javascript
search_contracts({
  awarding_agency: "Department of Veterans Affairs",
  naics_code: ["541512", "541511", "541519"],
  set_aside_type: "SDVOSB",
  award_date_start: "2024-01-01",
  ordering: "-obligated",
  limit: 50
})
```

**Analysis Focus:**
1. **Contract Size Distribution:**
   - How many in A6's $2M-$25M sweet spot?
   - Any trends toward larger/smaller contracts?

2. **Competitor Tracking:**
   - Which SDVOSB vendors winning most?
   - Market share shifts?
   - New entrants?

3. **Capability Trends:**
   - Common keywords in descriptions
   - Emerging technology requirements
   - Methodology preferences (Agile vs Waterfall)

4. **Gap Analysis:**
   - Capabilities A6 has vs market needs
   - Emerging requirements A6 should build

**Export Strategy:** Use `export_format: "csv"` for spreadsheet analysis

### Pattern 3: Recompete Pipeline Search

**Purpose:** Identify expiring VA contracts for recompete opportunities

```javascript
search_contracts({
  awarding_agency: "Department of Veterans Affairs",
  naics_code: ["541512", "541511", "541519"],
  expiration_date_after: "YYYY-MM-DD",   // +1 month from today
  expiration_date_before: "YYYY-MM-DD",  // +6 months from today
  ordering: "-obligated",
  limit: 50
})
```

**Note:** Filter results client-side for $2M+ contracts (no `obligated_min` parameter exists).

**Analysis Workflow:**
1. Run query for 1-6 month and 6-12 month windows separately
2. For each contract expiring:
   - Identify incumbent vendor
   - Review contract history (modifications, extensions)
   - Analyze scope alignment with A6 capabilities
   - Assess incumbent satisfaction signals
   - Estimate recompete timeline
3. Flag high-priority recompetes for BD tracking
4. Monitor for RFP/RFQ publication via `search_opportunities`

**Red Flags to Watch:**
- Recent contract modifications (incumbent may be favored)
- 5+ year contracts with steady modifications (satisfied customer)
- Scope changes that don't align with A6 capabilities

**Opportunity Signals:**
- Incumbent performance issues (check past performance databases)
- Scope expansion beyond incumbent's core capabilities
- New accessibility or modernization requirements

### Pattern 4: Competitive Intelligence - Oddball Tracking

**Purpose:** Track primary competitor Oddball's recent VA contract awards

```javascript
search_contracts({
  vendor_uei: "MHNFNJ1J5RF5",  // Oddball UEI
  awarding_agency: "Department of Veterans Affairs",
  fiscal_year: 2025,
  ordering: "-award_date"
})
```

**Analysis Focus:**
1. **Contract Size Trends:**
   - Moving up-market or down-market?
   - Average contract size changes?

2. **NAICS Distribution:**
   - Capability expansion beyond 541512/541511?
   - New service areas?

3. **Set-Aside Patterns:**
   - SDVOSB vs full & open ratio
   - Sole source frequency

4. **SPRUCE Usage:**
   - How many awards through SPRUCE IDIQ?
   - Task order frequency and size?

5. **Capability Areas:**
   - New technology keywords
   - Overlap with A6 capabilities
   - Differentiation opportunities

**Competitor UEIs:**
- Oddball: `MHNFNJ1J5RF5`
- Coforma: `CWKEYCF97KB6`
- CivicActions: `XSZFHRS5QFL5`
- Ad Hoc: Use exact legal name in `vendor_name`

### Pattern 5: Teaming Partner Discovery

**Purpose:** Find certified small businesses for teaming arrangements and subcontracting opportunities

**Primary Query:**
```javascript
search_entities({
  socioeconomic: "SDVOSB",          // Target certification (SDVOSB, VOSB, 8A, WOSB, HUBZONE)
  naics_code: ["541512", "541511", "541519"],
  state: "VA",                       // Geographic preference
  limit: 25,
  shape: "uei,legal_business_name,cage_code,business_types,primary_naics,physical_address(*)"
})
```

**Use Cases:**
- Find SDVOSB/VOSB partners for VA proposals (1,633+ in VA alone)
- Identify 8A or WOSB subcontractors to meet socioeconomic goals
- Locate competitors in specific geographies (combine NAICS + state)
- Quick vendor lookups for due diligence (use `name` parameter)

**Advanced Techniques:**
- **Payload optimization:** Use `shape` parameter to reduce payload by ~50% (faster responses)
- **Multi-certification search:** `socioeconomic: "SDVOSB|VOSB"` (OR logic)
- **Lightweight lookups:** `{name: "Company Name", shape: "uei,legal_business_name,cage_code,business_types"}`
- **Geographic targeting:** Combine `state` and `naics_code` to find local competitors

**Performance Characteristics:**
- Filtered queries: 300-400ms typical
- Large unfiltered searches: up to 9s (64K+ SDVOSBs)
- **Best practice:** Always use filters (state, NAICS) for better performance

**Known Limitations:**
- `registration_status` and SAM dates return null (API data issue)
- Cannot filter by active registration status
- Name matching is imprecise for `name` parameter

**Example Workflows:**

**Workflow 1: VA Proposal Teaming**
```javascript
// Step 1: Find SDVOSB partners in VA with IT capabilities
search_entities({
  socioeconomic: "SDVOSB",
  naics_code: ["541512", "541511", "541519"],
  state: "VA",
  limit: 50
})

// Step 2: Get detailed profile for promising candidates
get_vendor_profile({
  uei: "[CANDIDATE_UEI]",
  include_history: true,
  history_limit: 10
})

// Step 3: Assess performance trends
get_vendor_metrics({
  uei: "[CANDIDATE_UEI]",
  period_grouping: "quarter",
  months: 24
})
```

**Workflow 2: Subcontracting Goal Compliance**
```javascript
// Find diverse small businesses for different goal categories
search_entities({
  socioeconomic: "8A|WOSB|HUBZONE",
  naics_code: ["541512"],
  state: "CA",  // Near A6 HQ in Chula Vista
  limit: 25
})
```

**Workflow 3: Competitive Landscape Mapping**
```javascript
// Find all SDVOSB competitors nationwide
search_entities({
  socioeconomic: "SDVOSB",
  naics_code: ["541512", "541511", "541519"],
  limit: 100,
  shape: "uei,legal_business_name,cage_code,state,primary_naics"
})
```

### Pattern 6: Competitive Metrics Comparison

**Purpose:** Track competitor performance trends and assess vendor health for teaming decisions

**Primary Query:**
```javascript
get_vendor_metrics({
  uei: "MHNFNJ1J5RF5",              // Competitor UEI (Oddball example)
  period_grouping: "quarter",        // month/quarter/year
  months: 48                         // 4 years of history (max 120)
})
```

**Use Cases:**
- **Competitive intelligence:** Compare A6 vs competitors quarterly spending trends (Oddball, Ad Hoc, Coforma)
- **Teaming partner assessment:** Assess partner stability via consistent revenue patterns (24-48 months recommended)
- **Pipeline health monitoring:** Track monthly activity to identify revenue patterns and dry periods
- **Growth analysis:** Year-over-year growth analysis for proposals and capability statements
- **Subcontracting insights:** Monitor subcontract activity to understand prime/sub relationship dynamics

**Period Grouping Guidance:**

| Grouping | Recommended Months | Use Case |
|----------|-------------------|----------|
| `month` | 12-24 | Pipeline health, seasonality analysis |
| `quarter` | 24-48 | Competitive comparison, growth trends |
| `year` | 48-120 | Long-term strategic analysis |

**Analysis Examples:**

**Competitive Intelligence Comparison:**
```javascript
// Step 1: Get A6 metrics
get_vendor_metrics({
  uei: "UP4JHKJQ9MQ7",  // A6
  period_grouping: "quarter",
  months: 48
})

// Step 2: Get Oddball metrics
get_vendor_metrics({
  uei: "MHNFNJ1J5RF5",  // Oddball
  period_grouping: "quarter",
  months: 48
})

// Step 3: Get Coforma metrics
get_vendor_metrics({
  uei: "CWKEYCF97KB6",  // Coforma
  period_grouping: "quarter",
  months: 48
})

// Analysis: Compare quarterly trends, growth rates, market share shifts
```

**Teaming Partner Stability Assessment:**
```javascript
// Assess potential partner's revenue consistency
get_vendor_metrics({
  uei: "[PARTNER_UEI]",
  period_grouping: "quarter",
  months: 24
})

// Look for:
// - Consistent quarterly revenue (stable partner)
// - Growing trend (healthy pipeline)
// - Volatile patterns (risk indicator)
// - Recent dry periods (capacity availability)
```

**Pipeline Health Monitoring:**
```javascript
// Monthly view for A6 pipeline analysis
get_vendor_metrics({
  uei: "UP4JHKJQ9MQ7",
  period_grouping: "month",
  months: 24
})

// Identify:
// - Seasonal patterns (Q4 spikes common in federal)
// - Pipeline gaps needing BD attention
// - Award frequency trends
```

**Data Interpretation:**

**What the metrics show:**
- `award_obligations`: Prime contract spending for the period
- `subaward_obligations`: Subcontract spending (when vendor is prime)
- Results only include periods with activity (not zero-filled)
- Each period shows cumulative obligations for that timeframe

**Red Flags for Teaming:**
- Erratic quarterly revenue (unpredictable partner)
- Declining trend over 8+ quarters (losing market position)
- No recent awards (>6 months gap)
- Heavy subcontracting dependency (may not be strong prime)

**Green Flags for Teaming:**
- Consistent quarterly revenue ($1M-$5M range typical for small businesses)
- Steady or growing trend
- Mix of prime and sub work
- Regular award cadence

**Performance Characteristics:**
- Response time: 330-440ms typical (very fast)
- No pagination needed (returns time series, not individual records)
- Results only include periods with activity

**Known Limitations:**
- CSV export parameter accepted but returns JSON format (API bug)
- Use JSON format only
- Maximum 120 months lookback
- No vendor name returned (must track UEI->name mapping separately)

**Integration with Other Patterns:**

**Full Vendor Intelligence Workflow:**
```javascript
// Step 1: Find entity
search_entities({
  name: "Oddball",
  shape: "uei,legal_business_name,cage_code"
})

// Step 2: Get comprehensive profile
get_vendor_profile({
  uei: "MHNFNJ1J5RF5",
  include_history: true,
  history_limit: 10
})

// Step 3: Get performance trends
get_vendor_metrics({
  uei: "MHNFNJ1J5RF5",
  period_grouping: "quarter",
  months: 48
})

// Step 4: Get recent contracts for detailed analysis
search_contracts({
  vendor_uei: "MHNFNJ1J5RF5",
  awarding_agency: "Department of Veterans Affairs",
  fiscal_year: 2025,
  ordering: "-award_date"
})
```

### Pattern 7: HHS Diversification Opportunities

**Purpose:** Identify HHS opportunities aligned with A6's proven capabilities

**Active Solicitations:**
```javascript
search_opportunities({
  agency: "Department of Health and Human Services",
  naics_code: ["541512", "541511", "541519"],
  set_aside_type: "SDVOSB",
  posted_date_after: "YYYY-MM-DD",
  active: true,
  ordering: "-posted_date",
  limit: 50
})
```

**HHS Forecasts (Pipeline Planning):**
```javascript
search_forecasts({
  agency: "HHS",
  naics_code: ["541512", "541511", "541519"],
  fiscal_year: 2025,
  status: "PUBLISHED",
  limit: 50
})
```

**Focus Areas (Use `query` Parameter):**
- Citizen-facing digital services
- Form digitization
- Healthcare.gov modernization
- Accessibility compliance (508, WCAG)
- Agile/HCD methodology

**Refinement Query Example:**
```javascript
search_opportunities({
  agency: "Department of Health and Human Services",
  naics_code: ["541512", "541511", "541519"],
  query: "healthcare portal",
  posted_date_after: "YYYY-MM-DD",
  active: true,
  limit: 50
})
```

**Analysis Strategy:**
1. Identify recurring buyers within HHS
2. Map to A6's Simpler Grants experience
3. Look for similar scope (forms, citizen services)
4. Build target account list for BD outreach

## Advanced Search Strategies

### Multi-Agency SDVOSB Contract Awards

**Purpose:** Find SDVOSB contract awards across all agencies for diversification intelligence

```javascript
search_contracts({
  naics_code: ["541512", "541511", "541519"],
  set_aside_type: "SDVOSB",
  award_date_start: "YYYY-MM-DD",
  ordering: "-obligated",
  limit: 100
})
```

**Then filter by agency:**
- Exclude VA (already monitoring)
- Focus on HHS, GSA, Education, Labor, SBA
- Look for citizen-facing digital services

**Note:** Filter results client-side for $2M+ contracts.

### Accessibility Market Scan

**Purpose:** Find accessibility compliance opportunities across government

```javascript
search_opportunities({
  naics_code: ["541512", "541511", "541519"],
  query: "accessibility",
  set_aside_type: "SDVOSB",
  posted_date_after: "YYYY-MM-DD",
  active: true,
  ordering: "-posted_date",
  limit: 50
})
```

**A6 Advantage:** Verified 508 compliance expertise through VA work

**Target Agencies:**
- Any agency with public-facing systems
- Education, Labor, SBA (citizen services)
- Healthcare agencies (HHS, VA synergy)

### Forms Digitization Market Scan

**Purpose:** Find form digitization projects government-wide

```javascript
search_opportunities({
  naics_code: ["541512", "541511", "541519"],
  query: "form digitization",
  posted_date_after: "YYYY-MM-DD",
  active: true,
  ordering: "-posted_date",
  limit: 50
})
```

**A6 Advantage:** VA forms expertise (10-10EZ, 21-526EZ, CHAMPVA)

**Target Agencies:**
- HHS (grants, Medicare applications)
- Education (student aid forms - FAFSA)
- SBA (business registration)
- Labor (unemployment systems)

### Mobile Application Development Scan

**Purpose:** Find mobile app opportunities across government

```javascript
search_opportunities({
  naics_code: ["541512", "541511", "541519"],
  query: "mobile application",
  set_aside_type: "SDVOSB",
  posted_date_after: "YYYY-MM-DD",
  active: true,
  ordering: "-posted_date",
  limit: 50
})
```

**A6 Advantage:** VA mobile app experience (1.3M users), React Native expertise

### Incumbent Analysis

**Purpose:** Analyze a specific contract for recompete opportunity

```javascript
// First, find contracts by searching with query parameter
search_contracts({
  query: "36C10B21D0001",  // Search for contract number in description/ID
  awarding_agency: "Department of Veterans Affairs",
  limit: 10
})

// Then, find all contracts for that incumbent
search_contracts({
  vendor_uei: "[INCUMBENT_UEI]",
  awarding_agency: "Department of Veterans Affairs",
  ordering: "-award_date",
  limit: 50
})
```

**Analysis:**
1. Incumbent's total VA portfolio
2. Performance trends (growing or shrinking?)
3. Contract modification history
4. Set-aside usage patterns
5. Capability areas
6. Potential vulnerabilities

### Subcontracting Opportunity Scan

**Purpose:** Find large platform deals for subcontracting

```javascript
search_contracts({
  awarding_agency: "Department of Veterans Affairs",
  naics_code: ["541512"],
  award_date_start: "YYYY-MM-DD",
  ordering: "-obligated",
  limit: 50
})
```

**Strategy:**
1. Filter results client-side for $100M+ contracts
2. Identify platform vendors (V3Gate, Four Points, MetGreen)
3. Analyze contract scope for UX/design components
4. Reach out for teaming/subcontracting
5. Position A6 for accessibility, HCD, VA.gov expertise

**Common Platform Deal Keywords:**
- Salesforce, ServiceNow, AWS
- Enterprise License Agreement (ELA)
- Cloud migration
- Platform implementation

## Filtering and Qualification

### Contract Size Filtering

**Important:** The API does NOT support `obligated_min` or `obligated_max` parameters.

**Workaround - Sort + Client-Side Filter:**

```javascript
// Find large contracts ($2M+) - sort largest first
search_contracts({
  awarding_agency: "Department of Veterans Affairs",
  naics_code: ["541512", "541511", "541519"],
  ordering: "-obligated",  // Largest first
  limit: 100,
  shape: "key,obligated,recipient(*),description"  // Minimize payload
})

// Find small business opportunities - sort smallest first
search_contracts({
  awarding_agency: "Department of Veterans Affairs",
  naics_code: ["541512", "541511", "541519"],
  ordering: "obligated",   // Smallest first
  limit: 100,
  shape: "key,obligated,recipient(*),description"
})
```

**Ordering Options:**
| Value | Result | Use Case |
|-------|--------|----------|
| `-obligated` | Largest contracts first | Finding prime opportunities, platform deals |
| `obligated` | Smallest contracts first | Small business opportunities in your size range |

**Tips:**
- Use `shape` parameter to reduce payload 60-85% when over-fetching
- Paginate with `cursor` to scan through larger result sets
- Combine with other filters (`naics_code`, `fiscal_year`, `set_aside_type`) to reduce total results before client filtering

**Target Size Ranges (for client-side filtering):**
- Minimum viable: $500K+
- Ideal range: $2M-$25M
- Stretch: $25M-$50M
- Platform deals (sub only): $100M+

### Date Range Filters

**For Contracts (Past Awards):**
```javascript
{
  award_date_start: "YYYY-MM-DD",  // Start of date range
  award_date_end: "YYYY-MM-DD"     // End of date range
}
```

**For Opportunities (Active Solicitations):**
```javascript
{
  posted_date_after: "YYYY-MM-DD",   // Posted on or after this date
  posted_date_before: "YYYY-MM-DD",  // Posted on or before this date
  response_deadline_after: "YYYY-MM-DD"  // Deadline on or after this date
}
```

**Fiscal Year (Both Tools):**
```javascript
{ fiscal_year: 2025 }
```

**Expiring Contracts (Recompete Pipeline):**
```javascript
{
  expiration_date_after: "YYYY-MM-DD",   // Expiring after this date
  expiration_date_before: "YYYY-MM-DD"   // Expiring before this date
}
```

### Set-Aside Filters

**SDVOSB Only:**
```javascript
{ set_aside_type: "SDVOSB" }
```

**Multiple Set-Asides (OR logic):**
```javascript
{ set_aside_type: "SDVOSB|VOSB" }  // Pipe-separated for OR
```

**Full & Open (No Set-Aside):**
```javascript
{ set_aside_type: "NONE" }
```

**Note:** The `set_aside` field is often null in results. Don't rely on it for filtering.

### PSC Code Filters

**A6 Core Work:**
```javascript
{ psc_code: "DA01" }  // or ["DA01", "D302", "R408"] for multiple
```

**Avoid These PSC Codes:**
- J063 (Physical security systems)
- Hardware procurement codes
- Facilities/construction codes

## Complex Multi-Step Research Workflows

### Workflow 1: New Agency Penetration Strategy

**Objective:** Break into a new agency (e.g., Department of Education)

**Step 1: Market Sizing (Past Awards)**
```javascript
search_contracts({
  awarding_agency: "Department of Education",
  naics_code: ["541512", "541511", "541519"],
  fiscal_year: 2024,
  ordering: "-obligated",
  limit: 100
})
```
Analyze: Total market size, top contractors, set-aside distribution

**Step 2: SDVOSB Opportunity Assessment**
```javascript
search_contracts({
  awarding_agency: "Department of Education",
  naics_code: ["541512", "541511", "541519"],
  set_aside_type: "SDVOSB",
  fiscal_year: 2024,
  ordering: "-obligated",
  limit: 50
})
```
Analyze: SDVOSB usage rate, contract sizes, scope alignment

**Step 3: Capability Alignment**
```javascript
search_contracts({
  awarding_agency: "Department of Education",
  naics_code: ["541512", "541511", "541519"],
  query: "digital services",
  fiscal_year: 2024,
  limit: 50
})
```
Analyze: Opportunities matching A6 capabilities

**Step 4: Competitor Tracking**
```javascript
search_contracts({
  vendor_uei: "MHNFNJ1J5RF5",  // Oddball - check if they're there
  awarding_agency: "Department of Education",
  fiscal_year: 2024
})
```
Analyze: Are competitors already there? What are they doing?

**Step 5: Relationship Mapping**
- Identify key contracting offices
- Find recurring buyers
- Map to A6's past performance gaps
- Develop partnership/teaming strategy

### Workflow 2: Competitor Deep Dive

**Objective:** Understand a competitor's full strategy

**Step 1: Overall Portfolio**
```javascript
search_contracts({
  vendor_uei: "[COMPETITOR_UEI]",
  ordering: "-obligated",
  limit: 100
})
```

**Step 2: Agency Distribution**
Analyze results by agency, identify concentration

**Step 3: Capability Analysis**
Extract keywords from contract descriptions, categorize capabilities

**Step 4: Set-Aside Strategy**
Analyze set-aside usage patterns (SDVOSB vs full & open)

**Step 5: Growth Trends**
```javascript
search_contracts({
  vendor_uei: "[COMPETITOR_UEI]",
  fiscal_year: 2024,
  ordering: "-award_date"
})
```
Compare to prior years, identify growth areas

**Step 6: Contract Vehicle Analysis**
Identify IDIQs/BPAs, task order frequency, vehicle leverage

### Workflow 3: Recompete Campaign Planning

**Objective:** Plan pursuit of a specific expiring contract

**Step 1: Contract History**
```javascript
search_contracts({
  query: "[TARGET_CONTRACT_NUMBER]",
  awarding_agency: "[AGENCY]",
  limit: 10
})
```
Get full contract details, incumbent, scope, value

**Step 2: Incumbent Portfolio Analysis**
```javascript
search_contracts({
  vendor_uei: "[INCUMBENT_UEI]",
  awarding_agency: "[AGENCY]",
  ordering: "-award_date",
  limit: 50
})
```
Understand incumbent's relationship with agency

**Step 3: Modification History**
Look for recent mods, option exercises, scope changes
(May require additional data source beyond Tango)

**Step 4: Similar Contract Analysis**
```javascript
search_contracts({
  awarding_agency: "[AGENCY]",
  naics_code: ["[SAME_NAICS]"],
  psc_code: "[SAME_PSC]",
  fiscal_year: 2024,
  limit: 50
})
```
Understand agency's buying patterns for this type of work

**Step 5: Past Performance Gap Analysis**
- What does RFP likely require?
- Does A6 have similar contracts?
- Need teaming for past performance?
- Need to build capabilities?

## Query Optimization Tips

### Performance Considerations

**Limit Usage:**
- Start with `limit: 50` for exploration
- Increase to `limit: 100` for comprehensive analysis
- Use `cursor` parameter for pagination on large result sets

**Ordering:**
- `-award_date` for most recent activity
- `-obligated` for largest contracts first
- `-posted_date` for newest opportunities

**Field Selection:**
- Use `shape` parameter to reduce payload size
- Example: `shape: "key,piid,description,obligated,recipient(*)"`

### Search Precision

**NAICS Priority Order:**
Use `["541512", "541511", "541519"]` (array format)
- Leads with most common VA code (541512)

**Query Parameter:**
- Use specific keywords: "VA.gov" not just "website"
- Keep queries simple (1-2 terms work best)
- API uses AND logic - all words must match

**Common Pitfalls:**
- Searching without NAICS filter (too broad)
- Using only 541511 at VA (misses 95% of contracts)
- Ignoring contract size (wastes time on $100K deals)
- Not filtering by date (old data not relevant)

**Best Practices:**
- Always include agency + NAICS + date range
- Use set-aside filter for focused results
- Filter client-side for contract size
- Sort by relevant field (date or obligated)
- Start broad, then narrow with additional filters

## Export and Analysis

### Data Export Strategy

**CSV Export:**
Add `export_format: "csv"` to any search query for spreadsheet-compatible output.

```javascript
search_contracts({
  awarding_agency: "Department of Veterans Affairs",
  naics_code: ["541512", "541511", "541519"],
  fiscal_year: 2024,
  limit: 100,
  export_format: "csv"
})
```

**For Tracking Dashboard:**
1. Weekly snapshot queries
2. Store results in database
3. Track trends over time
4. Alert on new opportunities matching profile

**For Competitive Intelligence:**
1. Monthly competitor portfolio snapshots
2. Compare month-over-month changes
3. Identify market share shifts
4. Track capability expansion

### Analysis Templates

**Opportunity Qualification Scorecard:**
- Agency alignment (VA=5, HHS=4, GSA=3, Other=1)
- Contract size fit ($2M-$25M = 5, $500K-$2M = 3, other = 1)
- Set-aside advantage (SDVOSB=5, SB=3, Full&Open=2)
- Capability match (High=5, Medium=3, Low=1)
- Competitive position (Few competitors=5, Many=2)
- Total score: 20-25 = PURSUE, 15-19 = MONITOR, <15 = PASS

**Competitor Tracking Template:**
- Vendor name, UEI
- Contract count (quarterly)
- Total obligated (quarterly)
- Agency distribution
- Capability areas
- Notable wins/losses
- Strategic changes

**Market Trend Analysis:**
- Time period
- Total contract count
- SDVOSB percentage
- Average contract size
- Top contractors
- Emerging capabilities
- Methodology trends (Agile vs Waterfall)
